<?php
session_start();
require_once '../config/db.php';
require_once '../includes/auth_check.php';
require_once '../includes/functions.php';

// Set JSON header
header('Content-Type: application/json');

// Check if user has permission to view supplier orders
$supplierPermissions = [
    'view_suppliers', 'manage_suppliers', 'add_supplier', 'edit_supplier', 'delete_supplier',
    'search_suppliers', 'process_supplier_payment', 'add_supplier_item'
];

if (!hasAnyPermission($supplierPermissions) && !$isSuperAdmin) {
    echo json_encode(['success' => false, 'message' => 'Access denied']);
    exit;
}

if (!isset($_GET['order_id']) || !is_numeric($_GET['order_id'])) {
    echo json_encode(['success' => false, 'message' => 'Invalid order ID']);
    exit;
}

$order_id = intval($_GET['order_id']);

try {
    // Get order details
    $stmt = $pdo->prepare("
        SELECT 
            so.id,
            so.supplier_id,
            so.order_date,
            so.expected_delivery_date,
            so.status,
            so.notes,
            so.total_amount,
            so.created_at,
            so.updated_at,
            s.name as supplier_name,
            s.contact_person,
            s.phone as supplier_phone,
            s.email as supplier_email,
            s.address as supplier_address,
            u.username as created_by_username
        FROM supplier_orders so
        JOIN suppliers s ON so.supplier_id = s.id
        JOIN users u ON so.created_by = u.id
        WHERE so.id = ?
    ");
    $stmt->execute([$order_id]);
    $order = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$order) {
        echo json_encode(['success' => false, 'message' => 'Order not found']);
        exit;
    }
    
    // Get order items
    $stmt = $pdo->prepare("
        SELECT 
            soi.id,
            soi.product_id,
            soi.quantity,
            soi.unit_price,
            soi.total_price,
            soi.notes,
            soi.received_quantity,
            soi.status,
            p.name as product_name,
            p.description as product_description,
            c.name as category_name
        FROM supplier_order_items soi
        JOIN products p ON soi.product_id = p.id
        LEFT JOIN categories c ON p.category_id = c.id
        WHERE soi.order_id = ?
        ORDER BY p.name
    ");
    $stmt->execute([$order_id]);
    $items = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Format dates for display
    $order['order_date'] = date('Y-m-d', strtotime($order['order_date']));
    if ($order['expected_delivery_date']) {
        $order['expected_delivery_date'] = date('Y-m-d', strtotime($order['expected_delivery_date']));
    }
    $order['created_at'] = date('Y-m-d H:i:s', strtotime($order['created_at']));
    if ($order['updated_at']) {
        $order['updated_at'] = date('Y-m-d H:i:s', strtotime($order['updated_at']));
    }
    
    // Calculate totals
    $total_ordered = 0;
    $total_received = 0;
    foreach ($items as $item) {
        $total_ordered += intval($item['quantity']);
        $total_received += intval($item['received_quantity']);
    }
    
    $order['total_ordered'] = $total_ordered;
    $order['total_received'] = $total_received;
    $order['item_count'] = count($items);
    
    echo json_encode([
        'success' => true,
        'order' => $order,
        'items' => $items
    ]);
    
} catch (Exception $e) {
    error_log("Error fetching order details for order_id {$order_id}: " . $e->getMessage());
    echo json_encode([
        'success' => false, 
        'message' => 'An error occurred while fetching order details: ' . $e->getMessage(),
        'error_details' => $e->getMessage()
    ]);
}
?> 